<?php

namespace App\Enums;

use App\Traits\EnumToArray;


/**
 * StockMovementType
 *
 * This enum defines the types of stock movements that can occur
 * in the warehouse/inventory system.
 *
 * Each type is used for tracking stock quantity changes
 * and auditing inventory history.
 */
enum StockMovementType: string
{

    use EnumToArray;

/**
     * Inbound:
     * Stock coming into the warehouse.
     * Example: receiving goods from suppliers, manufacturing, or returns from customers.
     */
    case Inbound = 'inbound';

/**
     * Outbound:
     * Stock leaving the warehouse.
     * Example: fulfilling customer orders, internal usage, or sending goods to another branch.
     */
    case Outbound = 'outbound';

/**
     * Adjustment:
     * Manual changes to stock levels.
     * Example: correcting inventory counts after physical stocktaking, fixing errors.
     */
    case Adjustment = 'adjustment';

/**
     * Reservation:
     * Stock temporarily held for an order or request.
     * Example: when a customer places an order, the stock is reserved to avoid overselling.
     */
    case Reservation = 'reservation';

/**
     * Reservation Release:
     * Reserved stock is released back into available inventory.
     * Example: order cancellation, payment failure, or release after timeout.
     */
    case ReservationRelease = 'reservation_release';

/**
     * Transfer Out:
     * Stock sent from one warehouse to another.
     * Example: moving goods from a central warehouse to a branch warehouse.
     */
    case TransferOut = 'transfer_out';

/**
     * Transfer In:
     * Stock received from another warehouse.
     * Example: goods arriving from a different branch or central warehouse.
     */
    case TransferIn = 'transfer_in';


    public function label(): string
    {
        return match ($this) {
            self::Inbound            => __('Inbound'),
            self::Outbound           => __('Outbound'),
            self::Adjustment         => __('Adjustment'),
            self::Reservation        => __('Reservation'),
            self::ReservationRelease => __('Reservation Release'),
            self::TransferOut        => __('Transfer Out'),
            self::TransferIn         => __('Transfer In'),
        };
    }
}
