<?php

namespace App\Http\Controllers\Admin;

use Throwable;
use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Auth\Events\Registered;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $query = User::query();
        $users = $query->with(['roles', 'wallet'])->latest()->paginate(30);

        return view('pages.admin.settings.users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('pages.admin.settings.users.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:255|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
        ]);


        $validatedData['username'] = $validatedData['phone'];
        $validatedData['password'] = Hash::make($validatedData['password']);

        $user = User::create($validatedData);

        event(new Registered($user));

        return redirect()->route('admin.settings.users.show', $user->id)->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('pages.admin.settings.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        return view('pages.admin.settings.users.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8',
        ]);

        if (!empty($validatedData['password'])) {
            $validatedData['password'] = Hash::make($validatedData['password']);
        } else {
            unset($validatedData['password']);
        }

        $validatedData['username'] = $validatedData['phone'];

        $user->update($validatedData);

        return redirect()->route('admin.settings.users.show', $user->id)->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        return back();
    }

    public function userRoleAssignForm($id)
    {
        $user  = User::find($id);
        $roles = Role::all()->pluck('name', 'id');
        $user_roles = collect($user->getRoleNames());

        return view(
            'pages.admin.settings.users.assign-role',
            [
                'id'       => $id,
                'user'       => $user,
                'roles'      => $roles,
                'user_roles' => $user_roles,
            ]
        );
    }
    public function userRoleAssigns(Request $request, $id)
    {

        try {
            if (!User::find($id)->syncRoles($request->roles)) {
                return back()->with('status_fail', 'عملیات افزودن دسترسی با خطا مواجه شد.');
            }
        } catch (Throwable $e) {
            return back()->withError($e->getMessage())->withInput();
        }

        return back()->with('status_success', 'عملیات با موفقیت انجام شد.');
    }
}
