<?php

namespace App\Http\Controllers\Faq;

use Illuminate\Http\Request;
use Artemis\Faq\Models\Answer;
use Artemis\Faq\Models\Category;
use Artemis\Faq\Models\Question;
use App\Http\Controllers\Controller;
use Artemis\Faq\Enums\QuestionStatusTypeEnum;

class QuestionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Question::query();

        if ($request->has('search')) {
            $query->where('title', 'like', '%' . $request->input('search') . '%');
        }
        if ($request->has('category')) {
            $query->where('category_id', $request->input('category'));
        }

        $questions = $query->paginate(30);

        return view('pages.faq.questions.index', compact('questions'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $statuses = QuestionStatusTypeEnum::cases();
        $categories = Category::all()->mapWithKeys(function ($category, $key) {
            return [
                $category->id =>  $category->name
            ];
        });
        return view('pages.faq.questions.create', compact('statuses', 'categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate(
            [
                'title' => 'required|string|max:255',
                'body' => 'required|string',
                'category_id' => 'required|integer|exists:faq_categories,id',
            ]
        );

        $validated['user_id'] = auth()->id();
        $validated['status'] = QuestionStatusTypeEnum::OPEN->value;
        $validated['views'] = 0;
        $validated['slug'] = generateSlug(Question::class, $validated['title']);

        $question = Question::create($validated);

        return redirect()->route('faq.questions.show', $question->id)
            ->with('success', __('Question created successfully.'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Question $question)
    {
        $question->load(['user', 'category', 'tags']);

        $answers = $question->answers()->with('user')->withVoteScore()->latest()->paginate(10);

        return view('pages.faq.questions.show', compact('question', 'answers'));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
    /**
     * store a newly created answer.
     */
    public function storeAnswer(Request $request, Question $question)
    {
        $validated = $request->validate(
            [
                'body' => 'required|string',
            ]
        );

        $validated['user_id'] = auth()->id();
        $validated['question_id'] = $question->id;

        $question->answers()->create($validated);

        return redirect()->route('faq.questions.show', $question->id)
            ->with('success', __('Answer created successfully.'));
    }

    /**
     * Vote for an answer.
     */
    public function voteAnswer(Request $request)
    {
        $validated = $request->validate(
            [
                'answer_id' => 'required|integer|exists:faq_answers,id',
                'vote' => 'required|in:up,down',
            ]
        );

        $answer = Answer::findOrFail($validated['answer_id']);

        $vote = $validated['vote'];

        $answer->votes()->updateOrCreate(
            ['user_id' => auth()->id()],
            ['vote_type' => $vote]
        );

        return back()->with('success', __('Your vote has been recorded.'));
    }

    /**
     * Mark an answer as the best answer.
     */
    public function markAsBestAnswer(Request $request, Question $question, Answer $answer)
    {

        if ($question->best_answer_id == $answer->id)
            $question->update(['best_answer_id' => null]);
        else
            $question->update(['best_answer_id' => $answer->id]);

        return redirect()->route('faq.questions.show', $question->id)
            ->with('success', __('Answer marked as best successfully.'));
    }

    public function storeAnswerComment(Request $request, Question $question, Answer $answer)
    {
        $validated = $request->validate(
            [
                'body' => 'required|string',
            ]
        );

        $validated['user_id'] = auth()->id();

        $answer->comments()->create($validated);

        return redirect()->route('faq.questions.show', $question->id)
            ->with('success', __('Comment created successfully.'));
    }
    public function storeAnswerFlag(Request $request, Question $question, Answer $answer)
    {
        $validated = $request->validate(
            [
                'reason' => 'required|string',
            ]
        );

        $validated['user_id'] = auth()->id();

        $answer->flags()->create($validated);

        return redirect()->route('faq.questions.show', $question->id)
            ->with('success', __('Flag created successfully.'));
    }
}
