<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Services\SweetAlert;
use Illuminate\Http\Request;
use App\Models\ProductInventory;
use Illuminate\Support\Facades\DB;
use App\Models\CategoryAttributeItem;
use App\Models\ProductInventoryVariant;
use App\Models\Traits\HasCodeGeneration;

class InventoryVariantController extends Controller
{

    /**
     * Display a listing of the resource.
     */
    public function index(ProductInventory $inventory)
    {
        $variants = ProductInventoryVariant::where('product_inventory_id', $inventory->id)->get();
        $product = $inventory->product;

        return view('pages.admin.shop.variant.index', compact('inventory', 'variants', 'product'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(ProductInventory $inventory)
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, ProductInventory $inventory)
    {
        $data = $request->validate([
            'attributes'       => ['required', 'array', 'min:1'],
            'attributes.*.id'  => ['required', 'integer'],   // CategoryAttribute id
            'attributes.*.value' => ['required', 'integer'], // CategoryAttributeItem id
            'price'            => ['nullable', 'numeric', 'min:0'],
            'inventory_count'  => ['required', 'integer', 'min:1'],
            'warehouse_id'     => ['required'],
        ]);

        return DB::transaction(function () use ($data, $inventory) {

            // 1) Build and validate (attribute_id => item_id) pairs strictly
            $combinationItems = [];
            foreach ($data['attributes'] as $row) {
                $attrId = (int) $row['id'];
                $itemId = (int) $row['value'];

                $pairOk = CategoryAttributeItem::query()
                    ->where('id', $itemId)
                    ->where('category_attribute_id', $attrId)
                    ->exists();

                if (!$pairOk) {
                    return back()->withErrors([
                        'attributes' => __("Selected item (:item) doesn't belong to attribute (:attr).", [
                            'item' => $itemId,
                            'attr' => $attrId
                        ])
                    ])->withInput();
                }

                $combinationItems[$attrId] = $itemId;
            }

            // 2) Normalize
            ksort($combinationItems); // stable JSON order
            $combinationKey = collect($combinationItems)->values()->sort()->implode('-');

            // 3) Friendly duplicate check (still keep DB unique as final guard)
            $exists = ProductInventoryVariant::query()
                ->where('product_inventory_id', $inventory->id)
                ->where('combination_key', $combinationKey)
                ->exists();

            if ($exists) {
                return back()->withErrors([
                    'attributes' => __('This variant combination already exists for this inventory.'),
                ])->withInput();
            }

            // 4) Create variant (codes auto-generated in model boot via HasCodeGeneration)
            $variant = new ProductInventoryVariant([
                'user_id'              => auth()->id(),
                'shop_id'              => auth()->user()->shop->id,
                'product_inventory_id' => $inventory->id,
                'combination_items'    => $combinationItems,     // <-- rely on $casts
                'combination_key'      => $combinationKey,
                'price_override'       => $data['price'] ?? null,
                'currency_code'        => $data['price'] ? 'IRR' : null,
                'status'               => 'active',
            ]);
            $variant->save();


            if (isset($data['inventory_count']) && $data['inventory_count'] > 0) {
                auth()->user()->shop->storeInventoryItem($data['warehouse_id'], $variant, $data['inventory_count']);
            }



            return redirect()
                ->route('admin.shop.inventory.variant.index', $inventory->id)
                ->with('sweetalert', SweetAlert::success(__('success'), __('variant created successfully')));
        });
    }
    /**
     * Display the specified resource.
     */
    public function show(ProductInventory $inventory, ProductInventoryVariant $productInventoryVariant)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ProductInventory $inventory, ProductInventoryVariant $variant)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ProductInventory $inventory, ProductInventoryVariant $variant)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProductInventory $inventory, ProductInventoryVariant $variant)
    {
        $variant->delete();

        return redirect()->route('admin.shop.inventory.variant.index', $inventory->id)
            ->with('sweetalert', SweetAlert::success(__('success'), __('variant deleted successfully')));
    }
}
