<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Enums\OrderStatusEnum;
use App\Traits\HasTrackingCode;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Order extends Model
{
    use HasTrackingCode;
    protected $fillable = [
        'uuid',
        'created_by_id',
        'created_by_type',
        'ordered_by_id',
        'ordered_by_type',
        'orderable_id',
        'orderable_type',
        'address_id',
        'delivery_time_id',
        'type',
        'tracking_code',
        'name',
        'phone',
        'email',
        'amount',
        'amount_with_discount',
        'discount',
        'coupon_id',
        'reseller',
        'status',
        'description',
    ];

    /**
     * The part below is for this model relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function deliveries(): HasMany
    {
        return $this->hasMany(Delivery::class);
    }

    public function metas(): HasMany
    {
        return $this->hasMany(OrderMeta::class);
    }

    public function payments(): MorphMany
    {
        return $this->morphMany(Payment::class, 'payable');
    }
    public function address(): BelongsTo
    {
        return $this->belongsTo(Address::class);
    }
    public function deliveryTime(): BelongsTo
    {
        return $this->belongsTo(DeliveryTime::class);
    }
    public function coupon(): BelongsTo
    {
        return $this->belongsTo(Coupon::class);
    }
    public function createdBy(): MorphTo
    {
        return $this->morphTo();
    }
    public function orderedBy(): MorphTo
    {
        return $this->morphTo();
    }
    public function orderable(): MorphTo
    {
        return $this->morphTo();
    }
    /**
     * The above is for this model relationships
     */


    /**
     * The below is for this model scopes
     */
    // Scope to filter orders created today
    public function scopeToday(Builder $query): Builder
    {
        return $query->whereDate('created_at', Carbon::today());
    }

    // Scope to filter orders created in the current month
    public function scopeThisMonth(Builder $query): Builder
    {
        return $query->whereBetween('created_at', [
            Carbon::now()->startOfMonth(),
            Carbon::now()->endOfMonth(),
        ]);
    }

    // Scope to filter orders awaiting payment
    public function scopeAwaitingPayment(Builder $query): Builder
    {
        return $query->where('status', OrderStatusEnum::AwaitingPayment);
    }
    public function scopeAwaitingFulfillment(Builder $query): Builder
    {
        return $query->where('status', OrderStatusEnum::AwaitingFulfillment);
    }
    public function scopeCompleted(Builder $query): Builder
    {
        return $query->where('status', OrderStatusEnum::Completed);
    }

    /**
     * Scope: orders that contain at least one item from the given shop.
     * Also eager-loads only those items (and their inventory) that belong to that shop.
     *
     * @param  Builder        $query
     * @param  int|array<int> $shopId  Single shop id or an array of ids
     * @return Builder
     */
    public function scopeForShop(Builder $query, int|array $shopId): Builder
    {
        $shopIds = (array) $shopId;

        return $query
            // Only orders that have at least one item whose inventory belongs to the shop(s)
            ->whereHas('items.inventory', function (Builder $q) use ($shopIds) {
                $q->whereIn('shop_id', $shopIds);
            })
            // Eager-load items filtered to the same shop(s) + their inventory
            ->with([
                'items' => function (Builder $q) use ($shopIds) {
                    $q->whereHas('inventory', function (Builder $iq) use ($shopIds) {
                        $iq->whereIn('shop_id', $shopIds);
                    });
                },
                'items.inventory',
            ]);
    }
    /**
     * The above is for this model scopes
     */


    public static function boot()
    {
        parent::boot();
        self::creating(function ($model) {
            $model->uuid = Str::uuid()->toString();
            $model->tracking_code = self::createTrackingCode();
        });
    }
}
