<?php
namespace App\Services;

/**
 * IpPanelClient
 *
 * A reusable PHP class to interact with the Ippanel SOAP SMS service.
 */
class IpPanelClient
{
    /** @var SoapClient */
    private $client;
    /** @var string */
    private $user;
    /** @var string */
    private $pass;

    /**
     * Constructor.
     *
     * @param string $user    Your Ippanel username
     * @param string $pass    Your Ippanel password
     * @param string $wsdlUrl WSDL URL (optional)
     */
    public function __construct(string $user, string $pass, string $wsdlUrl = 'http://ippanel.com/class/sms/wsdlservice/server.php?wsdl')
    {
        // Disable WSDL caching for fresh service definitions
        ini_set('soap.wsdl_cache_enabled', '0');
        $this->user   = $user;
        $this->pass   = $pass;
        $this->client = new \SoapClient($wsdlUrl);
    }

    /**
     * Internal wrapper for SOAP calls (methods expecting user/pass first).
     *
     * @param string $method SOAP method name
     * @param array  $args   Method-specific arguments (excluding user/pass)
     *
     * @return mixed The SOAP call result
     * @throws \Exception on SOAP fault
     */
    private function call(string $method, array $args = [])
    {
        array_unshift($args, $this->user, $this->pass);
        try {
            return call_user_func_array([
                $this->client,
                $method
            ], $args);
        } catch (\SoapFault $fault) {
            throw new \Exception($fault->faultstring);
        }
    }

    // --- Service methods ---

    public function getLines()
    {
        return $this->call('GetLines');
    }

    public function checkMessageStatus(string $messageId)
    {
        return $this->call('CheckMessage', [$messageId]);
    }

    public function getDelivery(string $uniqId)
    {
        return $this->call('GetDelivery', [$uniqId]);
    }

    public function getCredit()
    {
        return $this->call('GetCredit');
    }

    public function getUserTime()
    {
        return $this->call('GetUsertime');
    }

    public function getTicketList()
    {
        return $this->call('GetTicketList');
    }

    public function getTicketDetail(string $ticketId)
    {
        return $this->call('GetTicketDetail', [$ticketId]);
    }

    public function addTicket(
        string $subject,
        string $description,
        string $type,
        string $importance,
        string $smsNotification,
        string $fileUrl = ''
    ) {
        return $this->call('AddTicket', [
            $subject,
            $description,
            $type,
            $importance,
            $smsNotification,
            $fileUrl
        ]);
    }

    public function answerTicket(
        string $ticketId,
        string $description,
        string $fileUrl = ''
    ) {
        return $this->call('AnswerTicket', [
            $ticketId,
            $description,
            $fileUrl
        ]);
    }

    public function deletePeriodSms(string $bulkId)
    {
        return $this->call('deleteperiod', [$bulkId]);
    }

    // --- Additional methods ---

    /**
     * Fetch inbox list.
     *
     * @return mixed
     */
    public function inboxList()
    {
        return $this->call('inboxlist');
    }

    /**
     * Send a templated (pattern) SMS.
     *
     * @param string $fromNum     Sender number
     * @param array  $toNumbers   Recipient numbers
     * @param string $patternCode Pattern identifier
     * @param array  $inputData   Key-value replacements for template
     * @return mixed
     * @throws \Exception on SOAP fault
     */
    public function sendPatternSms(
        string $fromNum,
        array $toNumbers,
        string $patternCode,
        array $inputData
    ) {
        try {
            return $this->client->sendPatternSms(
                $fromNum,
                $toNumbers,
                $this->user,
                $this->pass,
                $patternCode,
                $inputData
            );
        } catch (\SoapFault $fault) {
            throw new \Exception($fault->faultstring);
        }
    }

    /**
     * Send a basic SMS.
     *
     * @param string $fromNum        Sender number
     * @param array  $toNumbers      Recipient numbers
     * @param string $messageContent Message text
     * @param string $time           Scheduled time (YYYY-MM-DD or empty for immediate)
     * @return mixed
     */
    public function sendSms(
        string $fromNum,
        array $toNumbers,
        string $messageContent,
        string $time = ''
    ) {
        try {
            return $this->client->SendSMS(
                $fromNum,
                $toNumbers,
                $messageContent,
                $this->user,
                $this->pass,
                $time,
                'send'
            );
        } catch (\SoapFault $fault) {
            throw new \Exception($fault->faultstring);
        }
    }

    /**
     * Send point-to-point SMS (distinct messages per recipient).
     *
     * @param string $fromNum            Sender number
     * @param array  $toNumbers          Recipient numbers
     * @param array  $messageContents    Array of messages matching recipients
     * @param string $time               Scheduled time (YYYY-MM-DD or empty)
     * @return mixed
     */
    public function sendPointToPoint(
        string $fromNum,
        array $toNumbers,
        array $messageContents,
        string $time = ''
    ) {
        try {
            return $this->client->SendSMS(
                $fromNum,
                $toNumbers,
                $messageContents,
                $this->user,
                $this->pass,
                $time,
                'pointtopoint'
            );
        } catch (\SoapFault $fault) {
            throw new \Exception($fault->faultstring);
        }
    }

    /**
     * Send a voice message.
     *
     * @param int    $repeat     Number of times to repeat (1-3)
     * @param array  $toNumbers  Recipient numbers
     * @param string $type       File type (e.g., 'mp3')
     * @param string $fileUrl    URL of audio file
     * @param string $time       Scheduled time (YYYY-MM-DD or empty)
     * @return mixed
     */
    public function sendVoice(
        int $repeat,
        array $toNumbers,
        string $type,
        string $fileUrl,
        string $time = ''
    ) {
        return $this->call('sendvoice', [
            $repeat,
            $toNumbers,
            $type,
            $fileUrl,
            $time
        ]);
    }
}
