<?php

namespace App\Services\Models;

use App\Models\Cart;
use App\Models\Coupon;
use App\Models\Address;
use App\Models\Product;
use App\Models\CartItem;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Database\Eloquent\Collection;

class CartService
{
    /**
     * this method will only retrieve the cart, not create a new one
     */
    public function getCart(): ?Cart
    {
        if (Auth::check()) {
            return Cart::where('user_id', Auth::id())->where('status', 'active')->first();
        }

        $cartId = Session::get('cart_id');
        return $cartId ? Cart::where('id', $cartId)->where('status', 'active')->first() : null;
    }


    public function getOrCreateCart(): Cart
    {
        $cart = $this->getCart();
        if ($cart) return $cart;

        $data = ['status' => 'active', 'currency' => 'IRR', 'session_id' => Str::uuid()];
        if (Auth::check()) {
            $data['user_id'] = Auth::id();
        }

        $cart = Cart::create($data);
        if (!Auth::check()) {
            Session::put('cart_id', $cart->id);
        }

        return $cart;
    }

    /**
     * Update the current cart's delivery address.
     *
     * This method ensures a valid cart exists (creates one if missing),
     * validates that the given address belongs to the authenticated user (if logged in),
     * updates the `address_id` field on the cart,
     * and returns the refreshed cart instance including related data.
     *
     * @param  int  $addressId  The ID of the address to attach to the current cart.
     * @return \App\Models\Cart|null  The updated cart instance, or null if creation failed.
     *
     * @throws \Exception If the address does not belong to the authenticated user.
     */
    public function updateCartAddress(int $addressId): ?Cart
    {
        // Ensure we have a valid cart (create if missing)
        $cart = $this->getOrCreateCart();

        // Optional: Validate the address belongs to the authenticated user
        if (Auth::check()) {
            $address = Address::where('id', $addressId)
                ->where('user_id', Auth::id())
                ->first();

            if (! $address) {
                throw new \Exception('Invalid or unauthorized address.');
            }
        }

        // Update the cart with the selected address
        $cart->address_id = $addressId;
        $cart->save();

        // Return the latest state of the cart including related models
        return $cart->fresh(['items', 'user']);
    }


    /**
     * Add a product (and optional variant/inventory/shop info) to the current cart.
     *
     * This method ensures a valid cart exists, locates or creates the corresponding
     * CartItem entry, and updates its quantity and pricing data.
     *
     * Handles guest and authenticated users, serializes complex variant/data fields,
     * and supports discount/coupon tracking at item level.
     *
     * @param  int         $productId   The product being added.
     * @param  int|null    $inventoryId The inventory record (warehouse stock) if applicable.
     * @param  int|null    $variantId   The variant ID (e.g., size, color).
     * @param  int|null    $shopId      The shop or vendor ID (for multi-shop systems).
     * @param  int         $quantity    Quantity to add (default = 1).
     * @param  array|null  $variant     Variant attributes (color, size, etc.).
     * @param  string|null $price       Price override (optional; defaults to product price).
     * @param  string|null $discount    Discount applied on this item (if any).
     * @param  string|null $coupon      Coupon code applied (if any).
     * @param  array|null  $options        Additional metadata or configuration for this line.
     *
     * @return void
     *
     * @throws \Exception If the product does not exist or cart cannot be created.
     */
    public function add(
        int $productId,
        ?int $inventoryId = null,
        ?int $variantId = null,
        ?int $shopId = null,
        int $quantity = 1,
        ?array $variant = [],
        ?string $price = null,
        ?string $discount = null,
        ?string $coupon = null,
        ?array $options = []
    ): void {
        // Ensure we have a valid cart
        $cart = $this->getOrCreateCart();

        // Confirm product exists
        $product = Product::find($productId);
        if (! $product) {
            throw new \Exception("Product #{$productId} not found.");
        }

        // If no price provided, use product's current price
        $price = $price ?? $product->price;

        // Retrieve or create the cart item
        $item = CartItem::firstOrNew([
            'cart_id'      => $cart->id,
            'product_id'   => $productId,
            'inventory_id' => $inventoryId,
            'variant_id'   => $variantId,
            'shop_id'      => $shopId,
        ]);

        // Update or initialize values
        $item->quantity = ($item->quantity ?? 0) + $quantity;
        $item->variant = $variant ? json_encode($variant) : null;
        $item->price = $price;
        $item->discount = $discount;
        $item->coupon = $coupon;
        $item->options = $options ? json_encode($options) : null;
        $item->total = ($price - ($discount ?? 0)) * $item->quantity;

        $item->save();

        // Optional: recalculate cart totals
        $cart->load('items');
        $cart->recalculateTotals();
    }


    /**
     * Apply or update a coupon code on the current cart.
     *
     * This method ensures a valid cart exists (creates one if missing),
     * validates the coupon (if the system has a Coupon model),
     * assigns the coupon code to the cart, and recalculates totals.
     *
     * @param  string  $coupon  The coupon code entered by the user.
     * @return void
     *
     * @throws \Exception If the coupon is invalid or expired.
     */
    public function updateCoupon(string $coupon, $discount = 0): void
    {
        // Ensure we have a valid cart
        $cart = $this->getOrCreateCart();


        $couponModel = Coupon::where('code', $coupon)
            ->first();

        if (! $couponModel) {
            throw new \Exception("Invalid coupon code: {$coupon}");
        }


        // Update the coupon code on the cart
        $cart->coupon = $coupon;
        $cart->discount_total = $discount;

        $cart->save();

        $cart->recalculateTotals();
    }


    /**
     * Update the quantity of a product (optionally by variant/inventory/shop) in the current cart.
     *
     * If the provided quantity is zero or less, the item is removed from the cart.
     * Otherwise, the method updates the item quantity, ensures the cart exists,
     * and recalculates totals.
     *
     * @param  int  $productId   The product to update.
     * @param  int  $quantity    New quantity value (0 or less removes item).
     * @param  int|null  $inventoryId  Optional inventory ID (for warehouse or stock location).
     * @param  int|null  $variantId    Optional variant ID (e.g., size, color).
     * @param  int|null  $shopId       Optional shop/vendor ID (for multi-shop systems).
     * @return void
     *
     * @throws \Exception If cart does not exist or item is invalid.
     */
    public function updateQuantity(
        int $productId,
        int $quantity,
        ?int $inventoryId = null,
        ?int $variantId = null,
        ?int $shopId = null,
    ): void {
        // Ensure we have a valid cart (create if missing)
        $cart = $this->getOrCreateCart();

        // Base query builder for locating the item
        $query = CartItem::where([
            'cart_id'    => $cart->id,
            'product_id' => $productId,
        ])
            ->when(isset($inventoryId), fn($query) => $query->where('inventory_id', $inventoryId))
            ->when(isset($variantId), fn($query) => $query->where('variant_id', $variantId))
            ->when(isset($shopId), fn($query) => $query->where('shop_id', $shopId));

        // Remove item if quantity is zero or less
        if ($quantity <= 0) {
            $query->delete();
            $cart->load('items');
            $cart->recalculateTotals();
            return;
        }

        // Update quantity if item exists
        $item = $query->first();
        if (! $item) {
            throw new \Exception("Cart item not found for product #{$productId}.");
        }

        $item->quantity = $quantity;
        $item->total = ($item->price - ($item->discount ?? 0)) * $quantity;
        $item->save();

        // Recalculate totals after update
        $cart->load('items');
        $cart->recalculateTotals();
    }

    /**
     * Update (or remove) a specific cart item with new data such as quantity,
     * variant, price, discount, and coupon.
     *
     * If the given quantity is zero or less, the item is removed from the cart.
     * Otherwise, it creates or updates the item and recalculates totals.
     *
     * @param  int         $productId   The ID of the product being updated.
     * @param  int|null    $inventoryId The inventory record (warehouse stock) if applicable.
     * @param  int|null    $variantId   The variant ID (e.g., color, size).
     * @param  int|null    $shopId      The vendor/shop ID for multi-store setups.
     * @param  int         $quantity    The desired quantity (0 = remove).
     * @param  array       $variant     Variant attributes as array (color, size, etc.).
     * @param  string|null $price       Price override (defaults to product price if null).
     * @param  string|null $discount    Per-item discount (if applicable).
     * @param  string|null $coupon      Coupon code applied to this item (optional).
     * @param  array       $options        Additional metadata for this line (e.g. engraving, note).
     * @return void
     *
     * @throws \Exception If the product does not exist or the cart is invalid.
     */
    public function update(
        int $productId,
        ?int $inventoryId = null,
        ?int $variantId = null,
        ?int $shopId = null,
        int $quantity = 1,
        array $variant = [],
        ?string $price = null,
        ?string $discount = null,
        ?string $coupon = null,
        array $options = []
    ): void {
        // Ensure we have a valid cart
        $cart = $this->getOrCreateCart();

        // Validate product existence
        $product = Product::find($productId);
        if (! $product) {
            throw new \Exception("Product #{$productId} not found.");
        }

        // Remove item if quantity is zero or less
        if ($quantity <= 0) {
            CartItem::where([
                'cart_id'      => $cart->id,
                'product_id'   => $productId,
                'inventory_id' => $inventoryId,
                'variant_id'   => $variantId,
                'shop_id'      => $shopId,
            ])->delete();

            $cart->load('items');
            $cart->recalculateTotals();
            return;
        }

        // Determine price (prefer manual override, otherwise product price)
        $price = $price ?? $product->price;

        // Prepare serialized values (if not auto-cast)
        $variantJson = !empty($variant) ? json_encode($variant) : null;
        $optionsJson = !empty($options) ? json_encode($options) : null;

        // Create or update the cart item
        $item = CartItem::updateOrCreate(
            [
                'cart_id'      => $cart->id,
                'product_id'   => $productId,
                'inventory_id' => $inventoryId,
                'variant_id'   => $variantId,
                'shop_id'      => $shopId,
            ],
            [
                'quantity'  => $quantity,
                'variant'   => $variantJson,
                'price'     => $price,
                'discount'  => $discount,
                'coupon'    => $coupon,
                'options'      => $optionsJson,
                'total'     => ($price - ($discount ?? 0)) * $quantity,
            ]
        );

        // Recalculate cart totals
        $cart->load('items');
        $cart->recalculateTotals();
    }



    /**
     * Remove a single product (and all its variants/inventories) from the cart.
     *
     * This method ensures a valid cart exists, removes all matching items for the given product,
     * and recalculates the cart totals afterward.
     *
     * @param  int  $productId  The ID of the product to remove from the cart.
     * @return void
     */
    public function remove(int $productId, ?int $inventoryId = null, ?int $variantId = null, ?int $shopId = null): void
    {
        $cart = $this->getOrCreateCart();

        // Delete all items for this product (including variants/inventories)
        CartItem::where('cart_id', $cart->id)
            ->where('product_id', $productId)
            ->when(isset($inventoryId), fn($query) => $query->where('inventory_id', $inventoryId))
            ->when(isset($variantId), fn($query) => $query->where('variant_id', $variantId))
            ->when(isset($shopId), fn($query) => $query->where('shop_id', $shopId))
            ->delete();

        // Refresh and recalculate totals
        $cart->load('items');
        $cart->recalculateTotals();
    }


    /**
     * Completely clear the current cart, removing all items.
     *
     * This method ensures a valid cart exists, deletes all related items,
     * resets totals, and optionally removes any applied coupon or discounts.
     *
     * @return void
     */
    public function clear(): void
    {
        $cart = $this->getCart();

        // If there's no existing cart, nothing to clear
        if (! $cart) {
            return;
        }

        // Delete all items related to this cart
        $cart->items()->delete();

        // Reset totals and coupon
        $cart->update([
            'subtotal' => 0,
            'discount_total' => 0,
            'tax_total' => 0,
            'shipping_total' => 0,
            'grand_total' => 0,
            'coupon' => null,
        ]);
    }


    /**
     * Retrieve all items currently in the user's or guest's cart.
     *
     * This method ensures a valid cart exists (creates one if missing),
     * loads all items with related product information,
     * and returns a collection of CartItem models.
     *
     * @return \Illuminate\Database\Eloquent\Collection|\App\Models\CartItem[]|null
     */
    public function items(): Collection|null
    {
        $cart = $this->getCart();
        if (! $cart) {
            return null;
        }

        return $cart->items()
            ->with(['product', 'inventory', 'shop']) // optional extra eager loads
            ->get();
    }


    /**
     * Get the total quantity of all items in the current cart.
     *
     * This sums the `quantity` field across all cart items.
     *
     * @return int  Total number of items in the cart.
     */
    public function count(): int
    {
        $cart = $this->getCart();

        if (! $cart) {
            return 0;
        }

        // Using the relationship prevents reloading items() multiple times
        return $cart->items()->count();
    }


    public function total(): float
    {
        return $this->getCart()?->subtotal ?? 0;
    }

    public function totalPayable(): float
    {
        return $this->getCart()?->grand_total ?? 0;
    }

    /**
     * Merge the guest cart (stored in session) into the user's cart after login.
     *
     * This method checks if a guest cart exists in the session,
     * then combines its items into the user's cart.
     * It ensures quantities are summed when the same product/variant exists.
     *
     * @return void
     */
    public function mergeGuestCartIntoUser(): void
    {
        if (! Auth::check()) {
            return;
        }

        $guestCartId = Session::pull('cart_id');
        if (! $guestCartId) {
            return;
        }

        // Prevent merging into the same cart twice (if already merged)
        $existing = Cart::where('user_id', Auth::id())->first();
        if ($existing && $existing->id === $guestCartId) {
            return;
        }

        $guestCart = Cart::with('items')->find($guestCartId);
        if (! $guestCart) {
            return;
        }

        $userCart = $this->getOrCreateCart();

        foreach ($guestCart->items as $item) {
            $existingItem = $userCart->items()
                ->where('product_id', $item->product_id)
                ->where('inventory_id', $item->inventory_id)
                ->where('variant_id', $item->variant_id)
                ->where('shop_id', $item->shop_id)
                ->first();

            if ($existingItem) {
                $existingItem->increment('quantity', $item->quantity);
            } else {
                $userCart->items()->create($item->toArray());
            }
        }

        $userCart->load('items');
        $userCart->recalculateTotals();

        $guestCart->delete();
    }
}
