<?php

namespace App\Services\SEO;

use Illuminate\Support\HtmlString;

/**
 * SchemaManager
 * -------------------------------------------
 * Responsible for managing JSON-LD structured data
 * for SEO (Product, Organization, Article, etc.)
 * -------------------------------------------
 * Example:
 *     schema()->add('Product', [...])
 *     schema()->add('Organization', [...])
 *     {!! schema()->render() !!}
 */
class SchemaManager
{
    /**
     * @var array
     */
    protected array $schemas = [];

    /**
     * Add a schema block (JSON-LD)
     *
     * @param  string  $type
     * @param  array   $data
     * @return $this
     */
    public function add(string $type, array $data): self
    {
        $schema = array_merge([
            '@context' => 'https://schema.org',
            '@type' => $type,
        ], $data);

        $this->schemas[] = $schema;
        return $this;
    }

    /**
     * Add a Product schema (with rating & offer support)
     *
     * @param  array  $data  expects keys: name, image, description, sku, brand, url, price, currency, stock, rating, review_count
     * @return $this
     */
    public function product(array $data): self
    {
        $schema = [
            'name' => $data['name'] ?? null,
            'image' => (array) ($data['image'] ?? []),
            'description' => $data['description'] ?? null,
            'sku' => $data['sku'] ?? null,
            'brand' => [
                '@type' => 'Brand',
                'name' => $data['brand'] ?? 'Unknown',
            ],
            'offers' => [
                '@type' => 'Offer',
                'url' => $data['url'] ?? url()->current(),
                'priceCurrency' => $data['currency'] ?? 'IRR',
                'price' => $data['price'] ?? 0,
                'availability' => ($data['stock'] ?? 0) > 0
                    ? 'https://schema.org/InStock'
                    : 'https://schema.org/OutOfStock',
            ],
        ];

        // Optional rating
        if (!empty($data['rating'])) {
            $schema['aggregateRating'] = [
                '@type' => 'AggregateRating',
                'ratingValue' => number_format((float) $data['rating'], 1),
                'reviewCount' => $data['review_count'] ?? 0,
            ];
        }

        return $this->add('Product', $schema);
    }

    /**
     * Add custom schema data
     */
    public function custom(array $schema): self
    {
        $this->schemas[] = $schema;
        return $this;
    }

    /**
     * Render all schemas as JSON-LD <script> blocks
     *
     * @return HtmlString
     */
    public function render(): HtmlString
    {
        $html = '';
        foreach ($this->schemas as $schema) {
            $html .= '<script type="application/ld+json">' .
                json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) .
                "</script>\n";
        }
        return new HtmlString($html);
    }

    /**
     * Automatically render when called as string
     */
    public function __toString(): string
    {
        return (string) $this->render();
    }
}
