<?php

namespace App\Services;

use App\Models\Shop;
use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\PendingRequest;
use App\Exceptions\Woocommerce\ShopCredentialMissing;

class WooCommerceService
{

    private string $ck;
    private string $cs;
    private string $baseUrl;
    private int $timeout = 30;

    public function __construct(Shop $shop)
    {
        if ($shop->consumer_key === null || $shop->consumer_secret === null || $shop->woo_url === null) {
            throw new ShopCredentialMissing('one the following data is missing: consumer_key, consumer_secret, woo_url');
        }

        $this->ck = decrypt($shop->consumer_key);

        $this->cs = decrypt($shop->consumer_secret);

        $this->baseUrl = $shop->woo_url;


    }

    public function systemStatus(): array
    {
        return $this->get('system_status');
    }
    public function products(): array
    {
        return $this->get('products');
    }
    public function product($productId): array
    {
        $endpoint = "products/{$productId}";

        return $this->get($endpoint);
    }
    public function productVariations($productId)
    {
        $endpoint = "products/{$productId}/variations";

        return $this->get($endpoint);
    }



    /* ---------------- Internals ---------------- */

    private function client(): PendingRequest
    {
        $client = Http::timeout($this->timeout)
            ->acceptJson()
            ->withHeaders(['User-Agent' => 'YourMarketplace/1.0']);

        if ($this->isHttps()) {
            // Recommended: HTTPS + Basic Auth
            $client = $client->withBasicAuth($this->ck, $this->cs);
        }
        return $client;
    }

    private function get(string $endpoint, array $query = []): array
    {
        $url = "{$this->baseUrl}/wp-json/wc/v3/{$endpoint}";

        // For plain HTTP (local dev), Woo often blocks BasicAuth.
        // Use query-string auth as a fallback.
        if (!$this->isHttps()) {
            $query = array_merge($query, [
                'consumer_key'    => $this->ck,
                'consumer_secret' => $this->cs,
            ]);
        }

        $resp = $this->client()->get($url, $query);

        if ($resp->failed()) {
            // Helpful message for debugging
            throw new \RuntimeException(sprintf(
                "Woo API error %s: %s",
                $resp->status(),
                $resp->body()
            ));
        }

        /** @var array $data */
        $data = $resp->json();
        return $data ?? [];
    }

    private function isHttps(): bool
    {
        return str_starts_with(strtolower($this->baseUrl), 'https://');
    }
}
