<?php

namespace App\Traits;

use App\Enums\StockMovementType;
use App\Models\ProductInventoryVariant;
use App\Models\StockMovement;
use App\Models\Warehouse;
use App\Models\WarehouseInventory;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use RuntimeException;

trait HasWarehouse
{
    /**
     * Add (or increase) inventory for a variant in a warehouse.
     * (From previous step—kept here for completeness.)
     */
    public function storeInventoryItem(
        int $warehouseId,
        ProductInventoryVariant $variant,
        int $quantity = 1
    ): WarehouseInventory {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be a positive integer.');
        }

        $warehouse = $this->findWarehouseOrFail($warehouseId);
        $this->assertSameShop($warehouse, $variant);

        return DB::transaction(function () use ($warehouse, $variant, $quantity) {
            $inventory = $this->lockInventoryRow($warehouse->id, $variant->id);

            if ($inventory) {
                $inventory->increment('qty_on_hand', $quantity);
                $inventory->refresh();
            } else {
                $inventory = $warehouse->inventories()->create([
                    'product_inventory_id'         => $variant->product_inventory_id,
                    'product_inventory_variant_id' => $variant->id,
                    'sku'                          => $variant->sku_code,
                    'qty_on_hand'                  => $quantity,
                    'qty_reserved'                 => 0,
                    'min_qty'                      => 0,
                    'max_qty'                      => 0,
                ]);
            }

            $this->logMovement($warehouse->id, $variant, StockMovementType::Inbound, $quantity);

            return $inventory;
        });
    }

    /**
     * Reserve inventory (hold) for an order or process.
     * Increases qty_reserved without changing qty_on_hand.
     */
    public function reserveInventoryItem(
        int $warehouseId,
        ProductInventoryVariant $variant,
        int $quantity,
        ?string $referenceType = 'order',
        ?int $referenceId = null
    ): WarehouseInventory {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be positive.');
        }

        $warehouse = $this->findWarehouseOrFail($warehouseId);
        $this->assertSameShop($warehouse, $variant);

        return DB::transaction(function () use ($warehouse, $variant, $quantity, $referenceType, $referenceId) {
            $inventory = $this->lockInventoryRowOrFail($warehouse->id, $variant);

            // Ensure enough free stock to reserve
            $free = $inventory->qty_on_hand - $inventory->qty_reserved;
            if ($free < $quantity) {
                throw new RuntimeException('Not enough available stock to reserve.');
            }

            $inventory->increment('qty_reserved', $quantity);

            $this->logMovement(
                $warehouse->id,
                $variant,
                StockMovementType::Reservation,
                $quantity,
                $referenceType,
                $referenceId
            );

            return $inventory->refresh();
        });
    }

    /**
     * Release a reservation (return reserved stock back to available).
     */
    public function releaseReservation(
        int $warehouseId,
        ProductInventoryVariant $variant,
        int $quantity,
        ?string $referenceType = 'order',
        ?int $referenceId = null
    ): WarehouseInventory {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be positive.');
        }

        $warehouse = $this->findWarehouseOrFail($warehouseId);
        $this->assertSameShop($warehouse, $variant);

        return DB::transaction(function () use ($warehouse, $variant, $quantity, $referenceType, $referenceId) {
            $inventory = $this->lockInventoryRowOrFail($warehouse->id, $variant);

            if ($inventory->qty_reserved < $quantity) {
                throw new RuntimeException('Cannot release more than reserved.');
            }

            $inventory->decrement('qty_reserved', $quantity);

            $this->logMovement(
                $warehouse->id,
                $variant,
                StockMovementType::ReservationRelease,
                $quantity,
                $referenceType,
                $referenceId
            );

            return $inventory->refresh();
        });
    }

    /**
     * Consume a reservation to ship/fulfill (reserved -> outbound).
     * Decreases qty_reserved and qty_on_hand equally.
     */
    public function consumeReservation(
        int $warehouseId,
        ProductInventoryVariant $variant,
        int $quantity,
        ?string $referenceType = 'order',
        ?int $referenceId = null
    ): WarehouseInventory {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be positive.');
        }

        $warehouse = $this->findWarehouseOrFail($warehouseId);
        $this->assertSameShop($warehouse, $variant);

        return DB::transaction(function () use ($warehouse, $variant, $quantity, $referenceType, $referenceId) {
            $inventory = $this->lockInventoryRowOrFail($warehouse->id, $variant);

            if ($inventory->qty_reserved < $quantity) {
                throw new RuntimeException('Insufficient reserved stock to consume.');
            }
            if ($inventory->qty_on_hand < $quantity) {
                throw new RuntimeException('Insufficient on-hand stock to ship.');
            }

            $inventory->decrement('qty_reserved', $quantity);
            $inventory->decrement('qty_on_hand', $quantity);

            $this->logMovement(
                $warehouse->id,
                $variant,
                StockMovementType::Outbound,
                $quantity,
                $referenceType,
                $referenceId
            );

            return $inventory->refresh();
        });
    }

    /**
     * Manual adjustment (positive or negative).
     * Positive: increase on_hand (e.g., stock take gain).
     * Negative: decrease on_hand (e.g., stock take loss, damage).
     */
    public function adjustInventory(
        int $warehouseId,
        ProductInventoryVariant $variant,
        int $delta,
        ?string $referenceType = 'adjustment',
        ?int $referenceId = null
    ): WarehouseInventory {
        if ($delta === 0) {
            throw new InvalidArgumentException('Adjustment delta cannot be zero.');
        }

        $warehouse = $this->findWarehouseOrFail($warehouseId);
        $this->assertSameShop($warehouse, $variant);

        return DB::transaction(function () use ($warehouse, $variant, $delta, $referenceType, $referenceId) {
            $inventory = $this->lockInventoryRowOrFail($warehouse->id, $variant);

            if ($delta < 0 && $inventory->qty_on_hand < abs($delta)) {
                throw new RuntimeException('Insufficient on-hand stock for negative adjustment.');
            }

            $inventory->qty_on_hand += $delta;
            $inventory->save();

            $this->logMovement(
                $warehouse->id,
                $variant,
                StockMovementType::Adjustment,
                $delta, // you may log negative qty for clarity
                $referenceType,
                $referenceId
            );

            return $inventory->refresh();
        });
    }

    /**
     * Transfer stock between warehouses (TransferOut -> TransferIn).
     * Does not use reservations; it moves on-hand.
     */
    public function transferBetweenWarehouses(
        int $fromWarehouseId,
        int $toWarehouseId,
        ProductInventoryVariant $variant,
        int $quantity,
        ?string $referenceType = 'transfer',
        ?int $referenceId = null
    ): array {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be positive.');
        }
        if ($fromWarehouseId === $toWarehouseId) {
            throw new InvalidArgumentException('Source and destination warehouses must differ.');
        }

        $from = $this->findWarehouseOrFail($fromWarehouseId);
        $to   = $this->findWarehouseOrFail($toWarehouseId);
        $this->assertSameShop($from, $variant);
        $this->assertSameShop($to, $variant);

        return DB::transaction(function () use ($from, $to, $variant, $quantity, $referenceType, $referenceId) {
            // OUT from source
            $fromInv = $this->lockInventoryRowOrFail($from->id, $variant);
            if ($fromInv->qty_on_hand < $quantity) {
                throw new RuntimeException('Insufficient on-hand stock to transfer.');
            }
            $fromInv->decrement('qty_on_hand', $quantity);
            $this->logMovement($from->id, $variant, StockMovementType::TransferOut, $quantity, $referenceType, $referenceId);

            // IN to destination (create if missing)
            $toInv = $this->lockInventoryRow($to->id, $variant->id);
            if ($toInv) {
                $toInv->increment('qty_on_hand', $quantity);
                $toInv->refresh();
            } else {
                $toInv = $to->inventories()->create([
                    'product_inventory_id'         => $variant->product_inventory_id,
                    'product_inventory_variant_id' => $variant->id,
                    'sku'                          => $variant->sku_code,
                    'qty_on_hand'                  => $quantity,
                    'qty_reserved'                 => 0,
                    'min_qty'                      => 0,
                    'max_qty'                      => 0,
                ]);
            }
            $this->logMovement($to->id, $variant, StockMovementType::TransferIn, $quantity, $referenceType, $referenceId);

            return [$fromInv->refresh(), $toInv];
        });
    }

    /* =======================
       Internal helpers
       ======================= */

    protected function findWarehouseOrFail(int $warehouseId): Warehouse
    {
        $w = Warehouse::query()->find($warehouseId);
        if (! $w) {
            throw (new ModelNotFoundException)->setModel(Warehouse::class, [$warehouseId]);
        }
        return $w;
    }

    protected function assertSameShop(Warehouse $warehouse, ProductInventoryVariant $variant): void
    {
        $variantShopId = optional($variant->inventory)->shop_id;

        if ((int) $warehouse->shop_id !== (int) $variantShopId) {
            throw new InvalidArgumentException('Variant shop and warehouse shop do not match.');
        }
    }

    /**
     * Lock inventory row (if exists) for update.
     */
    protected function lockInventoryRow(int $warehouseId, int $variantId): ?WarehouseInventory
    {
        return WarehouseInventory::query()
            ->where('warehouse_id', $warehouseId)
            ->where('product_inventory_variant_id', $variantId)
            ->lockForUpdate()
            ->first();
    }

    /**
     * Lock inventory row or fail if not found.
     */
    protected function lockInventoryRowOrFail(int $warehouseId, ProductInventoryVariant $variant): WarehouseInventory
    {
        $row = $this->lockInventoryRow($warehouseId, $variant->id);
        if (! $row) {
            throw new ModelNotFoundException("Inventory row not found for variant #{$variant->id} in warehouse #{$warehouseId}.");
        }
        return $row;
    }

    protected function logMovement(
        int $warehouseId,
        ProductInventoryVariant $variant,
        StockMovementType $type,
        int $qty,
        ?string $referenceType = null,
        ?int $referenceId = null,
        array $meta = []
    ): void {
        StockMovement::create([
            'warehouse_id'                  => $warehouseId,
            'product_inventory_id'          => $variant->product_inventory_id,
            'product_inventory_variant_id'  => $variant->id,
            'type'                          => $type->value,
            'qty'                           => $qty,
            'reference_type'                => $referenceType,
            'reference_id'                  => $referenceId,
            'meta'                          => array_merge(['by' => auth()->id()], $meta),
            'moved_at'                      => now(),
            'user_id'                       => auth()->id(),
        ]);
    }
}
