<?php

namespace Artemis\Faq\Http\Controllers\api\v1;

use Str;
use Artemis\Faq\Models\Tag;
use Illuminate\Http\Request;
use Artemis\Faq\Traits\Taggable;
use Artemis\Faq\Models\Taggable as TaggableModel;
use Illuminate\Http\JsonResponse;
use Illuminate\Database\Eloquent\Model;
use Artemis\Faq\Http\Resources\TagResource;
use Artemis\Faq\Http\Requests\TagStoreRequest;
use Artemis\Faq\Http\Requests\TagUpdateRequest;
use Artemis\Faq\Http\Resources\TaggableResource;
use Artemis\Faq\Http\Controllers\api\ApiController;
use Symfony\Component\HttpFoundation\Response as ResponseCode;

class TagController extends ApiController
{
    public function index(): JsonResponse
    {
        return $this->success(TagResource::collection(Tag::all()));
    }
    
    public function store(TagStoreRequest $request): JsonResponse
    {
        Tag::create([
            'name' => $request->name,
            'slug' => $request->slug ?? Str::slug($request->name),
        ]);
        
        return $this->success(status: ResponseCode::HTTP_CREATED);
    }
    
    public function show(Tag $tag, Request $request): JsonResponse
    {
        $include = $request->include ? explode(',', $request->include) : [];
        
        $data = [
            'tag' => new TagResource($tag),
        ];
        
        if (in_array('taggable', $include, true)) {
            $page = $this->getPage($request);
            $perPage = $this->getPerPage($request);
            
            $result = TaggableModel::where('tag_id', $tag->id)->paginate(perPage: $perPage, page: $page);
            
            $data['taggables'] = [
                    'data' => TaggableResource::collection($result),
                ] + $this->paginateResult($result);
        }
        
        return $this->success($data);
    }
    
    public function update(TagUpdateRequest $request, Tag $tag): JsonResponse
    {
        $tag->update([
            'name' => $request->name ?? $tag->name,
            'slug' => $request->slug ?? ($request->name ? Str::slug($request->name) : $tag->slug),
        ]);
        
        return $this->success();
    }
    
    public function destroy(Tag $tag): JsonResponse
    {
        $tag->delete();
        
        return $this->success();
    }
    
    public function addTag(Request $request, string $taggable, int $id): JsonResponse
    {
        /** @var Model|Taggable $taggableModel */
        $taggableModel = resolveModelClass($taggable, $id);
        
        $taggableModel->attachTags($request->tags);
        
        return $this->success();
    }
    
    public function removeTag(Request $request, string $taggable, int $id): JsonResponse
    {
        /** @var Model|Taggable $taggableModel */
        $taggableModel = resolveModelClass($taggable, $id);
        
        $taggableModel->attachTags($request->tags);
        
        return $this->success();
    }
    
    public function getTags(string $taggable, int $id): JsonResponse
    {
        /** @var Model|Taggable $taggableModel */
        $taggableModel = resolveModelClass($taggable, $id);
        
        return $this->success(TagResource::collection($taggableModel->tags()->get()));
    }
}
