<?php

namespace Artemis\Faq\Providers;

use Route;
use Illuminate\Support\ServiceProvider;

class FaqServiceProvider extends ServiceProvider
{
    /**
     * Register bindings in the container.
     */
    public function register(): void
    {
        $this->mergeConfigFrom(package_path('config/artemis.php'), 'artemis');
        
        $this->app->register(EventServiceProvider::class);
    }
    
    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        $this->registerTranslations();
        $this->registerMigrations();
        $this->registerRoutes();
        
        if ($this->app->runningInConsole()) {
            $this->registerPublishableResources();
        }
    }
    
    private function registerTranslations(): void
    {
        $this->loadTranslationsFrom(package_path('resources/lang'), 'artemis');
    }
    
    private function registerMigrations(): void
    {
        $this->loadMigrationsFrom(package_path('database/migrations'));
    }
    
    private function registerRoutes(): void
    {
        $this->registerApiRoutes();
    }
    
    private function registerApiRoutes(): void
    {
        Route::group($this->getApiRoutesConfigurations(), function () {
            $this->loadRoutesFrom(package_path('routes/api/api.php'));
        });
    }
    
    private function getApiRoutesConfigurations(): array
    {
        return [
            'prefix' => config('artemis.faq.api.prefix'),
            'middleware' => config('artemis.faq.api.middleware'),
        ];
    }
    
    private function registerPublishableResources(): void
    {
        $this->publishMigrations();
        $this->publishControllers();
        $this->publishProviders();
        $this->publishRequests();
        $this->publishRoutes();
        $this->publishResources();
    }
    
    private function publishMigrations(): void
    {
        $this->publishes([
            package_path('database/migrations') => database_path('migrations/artemis'),
        ], 'artemis-faq-migrations');
    }
    
    private function publishControllers(): void
    {
        $this->publishes([
            package_path('src/Http/Controllers') => app_path('Http/Controllers/artemis'),
        ], 'artemis-faq-controllers');
    }
    
    private function publishProviders(): void
    {
        $this->publishes([
            package_path('src/Providers') => app_path('Providers/artemis'),
        ], 'artemis-faq-providers');
    }
    
    private function publishRequests(): void
    {
        $this->publishes([
            package_path('src/Http/Requests') => app_path('Http/Requests/artemis'),
        ], 'artemis-faq-requests');
    }
    
    private function publishResources(): void
    {
        $this->publishes([
            package_path('src/Http/Resources') => app_path('Http/Resources/artemis'),
        ], 'artemis-faq-resources');
    }
    
    private function publishRoutes(): void
    {
        $this->publishApiRoutes();
    }
    
    private function publishApiRoutes(): void
    {
        $this->publishes([
            package_path('routes/api/api.php') => base_path('routes/api/artemis'),
        ], 'artemis-faq-api-routes');
    }
}
