# 🧬 Loyalty Trait Usage Guide

This guide explains how to use the trait-based features included in the `artemis/laravel-loyalty` package within your application's `User` model.

---

## 🧩 Available Traits

### 1. `HasLoyalty`

Tracks and manages loyalty point relationships and balance.

```php
use Artemis\Loyalty\Concerns\HasLoyalty;

class User extends Authenticatable
{
    use HasLoyalty;
}
```

**Key Methods:**

* `loyaltyPoints()` → relation to user's loyalty points
* `redemptions()` → relation to reward redemptions
* `loyalty_balance` → current sum of points (cached)
* `hasEnoughPoints($amount)` → checks if user has enough
* `addPoints($amount, $type = 'manual', $sourceId = null)`
* `spendPoints($amount)` → soft/hard deduction logic (customizable)

---

### 2. `InteractsWithTiers`

Automatically calculates and updates tier level.

```php
use Artemis\Loyalty\Concerns\InteractsWithTiers;

class User extends Authenticatable
{
    use InteractsWithTiers;
}
```

**Key Methods:**

* `currentTier()` → returns current Tier model based on points
* `upgradeTier()` → updates `tier_level` if needed

---

### 3. `InteractsWithReferrals`

Links users who referred or were referred.

```php
use Artemis\Loyalty\Concerns\InteractsWithReferrals;

class User extends Authenticatable
{
    use InteractsWithReferrals;
}
```

**Key Methods:**

* `referralsMade()` → users this user has referred
* `referredBy()` → user who referred this one
* `totalReferralPoints()` → total earned from referrals

---

## 🔁 Example: Composing All Traits

```php
use Artemis\Loyalty\Concerns\HasLoyalty;
use Artemis\Loyalty\Concerns\InteractsWithTiers;
use Artemis\Loyalty\Concerns\InteractsWithReferrals;

class User extends Authenticatable
{
    use HasLoyalty, InteractsWithTiers, InteractsWithReferrals;
}
```

---

## 💡 Recommendations

* Call `$user->upgradeTier()` after point changes to update tier.
* Use `$user->hasEnoughPoints($cost)` before redemptions.
* Group traits logically if extending with team/gamification behaviors later.

Let me know if you'd like a test suite, example controller, or Livewire/Blade UI around these methods.
