<?php

namespace Artemis\Loyalty\Concerns;

use Exception;
use Artemis\Loyalty\Models\Tier;
use Artemis\Loyalty\Models\Referral;
use Artemis\Loyalty\Events\ReferralMade;
use Artemis\Loyalty\Models\LoyaltySetting;
use Artemis\Loyalty\Exceptions\InvalidReferralCodeException;
use Artemis\Loyalty\Exceptions\UserAlreadyReferredException;

trait InteractsWithReferrals
{
    public function referralsMade()
    {
        return $this->hasMany(Referral::class, 'referrer_id');
    }

    public function referredBy()
    {
        return $this->hasOne(Referral::class, 'referee_id');
    }

    public function totalReferralPoints(): int
    {
        return $this->referralsMade()->sum('bonus_awarded');
    }

    public function referredMe(string $code): bool
    {
        $referrer = self::where('referral_code', $code)->first();

        if ($referrer) {

            // Check if the user has already been referred by someone
            if ($this->referredBy()->exists()) {
                throw new UserAlreadyReferredException('User has already been referred by someone.');
            }

            $referral = Referral::create([
                'referrer_id' => $referrer->id,
                'referee_id' => $this->id,
                'bonus_awarded' => LoyaltySetting::get('referral_bonus', 10),
            ]);

            event(new ReferralMade($referrer, $this, $referral->bonus_awarded));

            return true;
        }
        throw new InvalidReferralCodeException('Invalid referral code.');
    }
}
