<?php

namespace Artemis\Loyalty\Providers;

use Illuminate\Support\Facades\Blade;
use Illuminate\Support\ServiceProvider;
use Artemis\Loyalty\Console\Commands\PublishViewsCommand;
use Artemis\Loyalty\Console\Commands\LoyaltyTablesCommand;
use Artemis\Loyalty\Console\Commands\PublishControllersCommand;

class LoyaltyServiceProvider extends ServiceProvider
{
    /**
     * The path to the package's base directory.
     *
     * @var string
     */
    protected string $packageBasePath = __DIR__ . '/../..';

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register(): void
    {
        $this->mergeConfigFrom(
            $this->packageBasePath . '/config/loyalty.php',
            'loyalty'
        );

        $this->app->register(EventServiceProvider::class);
    }

    /**
     * Bootstrap any package services.
     *
     * @return void
     */
    public function boot(): void
    {
        $this->loadMigrations();
        $this->loadRoutes();
        $this->loadViews();
        $this->loadCommands();
        $this->registerBladeComponents();

        // Register publishable assets
        $this->registerPublishableAssets();
    }

    /**
     * Register all publishable package assets.
     *
     * @return void
     */
    protected function registerPublishableAssets(): void
    {
        if ($this->app->runningInConsole()) {
            // Publish config file
            $this->publishes([
                $this->packageBasePath . '/config/loyalty.php' => config_path('loyalty.php'),
            ], 'loyalty-config');

            // Publish views
            $this->publishes([
                $this->packageBasePath . '/resources/views' => resource_path('views/vendor/loyalty'),
            ], 'loyalty-views');

            // Publish migrations
            $this->publishes([
                $this->packageBasePath . '/database/migrations' => database_path('migrations'),
            ], 'loyalty-migrations');

            // Publish assets (JS, CSS, images)
            $this->publishes([
                $this->packageBasePath . '/resources/assets' => public_path('vendor/loyalty'),
            ], 'loyalty-assets');

            // Publish translations
            $this->publishes([
                $this->packageBasePath . '/resources/lang' => lang_path('vendor/loyalty'),
            ], 'loyalty-translations');

            // Publish all assets with one tag
            $this->publishes([
                $this->packageBasePath . '/config/loyalty.php' => config_path('loyalty.php'),
                $this->packageBasePath . '/resources/views' => resource_path('views/vendor/loyalty'),
                $this->packageBasePath . '/database/migrations' => database_path('migrations'),
                $this->packageBasePath . '/resources/assets' => public_path('vendor/loyalty'),
                $this->packageBasePath . '/resources/lang' => lang_path('vendor/loyalty'),
            ], 'loyalty-all');
        }
    }

    /**
     * Load and register package migrations.
     *
     * @return void
     */
    protected function loadMigrations(): void
    {
        if ($this->app->runningInConsole()) {
            $this->loadMigrationsFrom($this->packageBasePath . '/database/migrations');
        }
    }

    /**
     * Load and register package routes.
     *
     * @return void
     */
    protected function loadRoutes(): void
    {
        $this->loadRoutesFrom($this->packageBasePath . '/routes/api.php');
        $this->loadRoutesFrom($this->packageBasePath . '/routes/web.php');
        $this->loadRoutesFrom($this->packageBasePath . '/routes/admin.php');
    }

    /**
     * Load and register package views.
     *
     * @return void
     */
    protected function loadViews(): void
    {
        $this->loadViewsFrom($this->packageBasePath . '/resources/views', 'loyalty');
    }

    /**
     * Register package commands.
     *
     * @return void
     */
    protected function loadCommands(): void
    {
        if ($this->app->runningInConsole()) {
            $this->commands([
                PublishControllersCommand::class,
                LoyaltyTablesCommand::class,
                PublishViewsCommand::class,
            ]);
        }
    }

    /**
     * Register Blade components.
     *
     * @return void
     */
    protected function registerBladeComponents(): void
    {
        Blade::componentNamespace('Artemis\Loyalty\View\Components', 'loyalty');
    }
}
